"""##################################### TERMS OF USE ###########################################
# The following code is provided for demonstration purpose only, and should not              #
# be used without independent verification. Recorded Future makes no representations         #
# or warranties, express, implied, statutory, or otherwise, regarding any aspect of          #
# this code or of the information it may retrieve, and provides it both strictly “as-is”     #
# and without assuming responsibility for any information it may retrieve. Recorded Future   #
# shall not be liable for, and you assume all risk of using, the foregoing. By using this    #
# code, Customer represents that it is solely responsible for having all necessary licenses, #
# permissions, rights, and/or consents to connect to third party APIs, and that it is solely #
# responsible for having all necessary licenses, permissions, rights, and/or consents to     #
# any data accessed from any third party API.                                                #
##############################################################################################
"""

import logging
import os
import re
from datetime import datetime, timedelta
from json.decoder import JSONDecodeError

import requests
from psengine.collective_insights import RFInsight
from requests import ConnectionError, ConnectTimeout, HTTPError, ReadTimeout

from .constants import (
    API_BASE_URL_MAP,
    COLLECTIVE_INSIGHTS_INCIDENT_NAME,
    COLLECTIVE_INSIGHTS_INCIDENT_TYPE,
    EXABEAM_AUTH_API,
    EXABEAM_EVENT_FIELDS,
    EXABEAM_SEARCH_API,
    IPV6_REGEX,
    LOOKBACK,
    MAX_RESULTS,
    MITRE_REGEX,
    TIMESTAMP_FORMAT,
)
from .errors import CategoryParseException, ExabeamError, RFScriptError, WriteFileError

TIMESTAMP_FILE = os.path.join('config', 'latest_timestamp.txt')


class ExabeamCI:
    """Exabeam for Collective Insights class"""

    def __init__(self, region: str, client_id: str, client_secret: str, lookback: int = LOOKBACK):
        self.log = logging.getLogger('psengine')
        self.client_id = client_id
        self.client_secret = client_secret
        self.lookback = lookback
        self.latest_timestamp = datetime.now().strftime(TIMESTAMP_FORMAT)
        self.base_url = self._set_api_base_url(region)
        self._validate_auth()
        self.auth_url = self.base_url + EXABEAM_AUTH_API
        self.search_url = self.base_url + EXABEAM_SEARCH_API
        self.exabeam_oauth_token = self._fetch_exabeam_oauth_token()
        self.start_timestamp = self._set_start_timestamp(
            lookback=lookback, timestamp_file=TIMESTAMP_FILE
        )

    def _set_api_base_url(self, region: str):
        """Returns Exabeam API base url based on instance region"""
        return API_BASE_URL_MAP.get(region)

    def _validate_auth(self):
        if self.base_url is None:
            self.log.error("API Base URL is not set. Rerun script with 'region' specified")
            raise RFScriptError
        if self.client_id is None:
            self.log.error("Exabeam Client ID not set. Rerun script with 'client_id' specified")
            raise RFScriptError
        if self.client_secret is None:
            self.log.error(
                "Exabeam Client Secret is not set. Rerun script with 'client_secret' specified"
            )
            raise RFScriptError

    def _fetch_exabeam_oauth_token(self) -> str:
        """Gets Exabeam OAuth response from client_id and client_secret

        Args:
            base_url (str): Exabeam API base url
            client_id (str): Exabeam API Client ID
            client_secret (str): Exabeam API Client Secret

        Returns:
            oauth (str): Exabeam OAuth token
        """
        payload = {
            'grant_type': 'client_credentials',
            'client_id': self.client_id,
            'client_secret': self.client_secret,
        }
        headers = {'accept': 'application/json', 'content-type': 'application/json'}

        try:
            response = requests.post(self.auth_url, json=payload, headers=headers)
            response.raise_for_status()
            oauth = response.json()['access_token']
        except (HTTPError, ConnectTimeout, ConnectionError, ReadTimeout) as err:
            self.log.error(f'Error authenticating to Exabeam: {err}')
            raise ExabeamError(err)
        except KeyError as err:
            self.log.error(f"Unable to read 'access_token' from Exabeam API response: {err}")
            raise ExabeamError(err)
        return oauth

    def _set_start_timestamp(self, lookback: int, timestamp_file: str = TIMESTAMP_FILE) -> str:
        """Reads most recent timestamp from file if it exists
        Otherwise sets lookback from script argument

        Args:
            lookback (int): Lookback hours provided from script input
            timestamp_file (str): File for saved timestamp

        Returns:
            start_timestamp (dict): time period used to query Exabeam alerts
        """
        if os.path.exists(timestamp_file):
            self.log.info(f'Reading latest alert timestamp from {timestamp_file}')
            try:
                with open(timestamp_file, 'r') as f:
                    latest_timestamp = f.read()
                timestamp = datetime.strptime(latest_timestamp, TIMESTAMP_FORMAT)
                updated_timestamp = timestamp + timedelta(milliseconds=1)
                lookback_str = updated_timestamp.strftime(TIMESTAMP_FORMAT)
                self.log.info(f'Using latest alert timestamp from file: {lookback_str}')
                start_timestamp = lookback_str
            except OSError as err:
                self.log.error(f'Error reading latest timestamp from file: {err}')
                self.log.info(f"Using '{lookback}' hour(s) instead")
                start_datetime = datetime.strptime(self.latest_timestamp, TIMESTAMP_FORMAT)
                start_timestamp = (start_datetime - timedelta(hours=lookback)).strftime(
                    TIMESTAMP_FORMAT
                )
        else:
            self.log.info(
                f"File '{timestamp_file}' does not exist. Using '{lookback}' hour(s) instead"
            )
            start_datetime = datetime.strptime(self.latest_timestamp, TIMESTAMP_FORMAT)
            start_timestamp = (start_datetime - timedelta(hours=lookback)).strftime(
                TIMESTAMP_FORMAT
            )
        return start_timestamp

    def fetch_exabeam_events(self) -> list:
        """Gets Exabeam events that have been generated since lookback time period

        Returns:
            result (list): list of Exabeam Events
        """
        payload = {
            'limit': MAX_RESULTS,
            'distinct': False,
            'fields': EXABEAM_EVENT_FIELDS,
            'startTime': self.start_timestamp,
            'endTime': self.latest_timestamp,
            'filter': 'is_ioc:true',
        }
        headers = {
            'accept': 'application/json',
            'content-type': 'application/json',
            'authorization': f'Bearer {self.exabeam_oauth_token}',
        }
        try:
            response = requests.post(self.search_url, json=payload, headers=headers)
            response.raise_for_status()
            result = response.json()['rows']
        except (HTTPError, ConnectTimeout, ConnectionError, ReadTimeout) as err:
            self.log.error(f'Error querying Exabeam events: {err}')
            raise ExabeamError(err)
        except JSONDecodeError as err:
            self.log.error(f'Error parsing Exabeam events response as json: {err}')
            raise ExabeamError(err)
        except KeyError as err:
            self.log.error(f'Error parsing Exabeam events response: {err}')
            raise ExabeamError(err)
        return result

    def fetch_exabeam_event_details(self, exabeam_events: list) -> list:
        """Gets Exabeam ioc, malware, and/or mitre fields for each event

        Args:
            base_url (str): Exabeam API base url
            exabeam_oauth_token (str): Exabeam API OAuth Token
            exabeam_events (list): Exabeam events
            time_filters (dict): Start and end times for event query

        Returns:
            results (list): list of Exabeam Events
        """
        results = []
        headers = {
            'accept': 'application/json',
            'content-type': 'application/json',
            'authorization': f'Bearer {self.exabeam_oauth_token}',
        }

        for event in exabeam_events:
            event_id = event.get('id')
            ioc_fields = event.get('ioc_fields')
            groups = ['mitre_labels', 'malware_family']
            event_fields = EXABEAM_EVENT_FIELDS + ioc_fields + groups
            payload = {
                'limit': 1,
                'distinct': False,
                'fields': event_fields,
                'startTime': self.start_timestamp,
                'endTime': self.latest_timestamp,
                'filter': f'id:"{event_id}"',
            }

            try:
                response = requests.post(self.search_url, json=payload, headers=headers)
                response.raise_for_status()
                exabeam_event = response.json()['rows'][0]
            except (HTTPError, ConnectTimeout, ConnectionError, ReadTimeout) as err:
                self.log.error(f'Error querying Exabeam events: {err}')
                raise ExabeamError(err)
            except JSONDecodeError as err:
                self.log.error(f'Error parsing Exabeam events response as json: {err}')
                raise ExabeamError(err)
            except (KeyError, IndexError) as err:
                self.log.error(f'Error parsing Exabeam events response: {err}')
                raise ExabeamError(err)
            results.append(exabeam_event)
        return results

    def format_exabeam_events(self, exabeam_events: list) -> list:
        """Parses Exabeam events data into Recorded Future Collective Insights format

        Args:
            exabeam_events (list): Exabeam API base url

        Returns:
            collective_insights_data (list): list of formatted Exabeam events
        """
        self.log.info('Preparing Exabeam events for Collective Insights submission')
        try:
            latest_event = sorted(
                exabeam_events,
                key=lambda x: datetime.strptime(x['time'], TIMESTAMP_FORMAT[:-1]),
                reverse=True,
            )[0]
            self.latest_timestamp = latest_event['time'] + 'Z'
        except (KeyError, IndexError):
            self.log.warning(
                'Did not parse latest timestamp from document, defaulting to execution time'
            )

        collective_insights_data = []

        for event in exabeam_events:
            try:
                event_id = event['id']
                event_timestamp = event['time']
                event_ioc_fields = event.get('ioc_fields', [])
                mitre_codes = event.get('mitre_labels')
                malwares = event.get('malware_family')
                for ioc_field in event_ioc_fields:
                    event_ioc = event.get(ioc_field)
                    try:
                        event_ioc_type = self.get_category(event_ioc)
                    except CategoryParseException as err:
                        self.log.error(err)
                        self.log.error(
                            f'Not able to parse indicator type for {event_ioc}. Skipping'
                        )
                        continue
                    insight_data = {
                        'ioc_value': event_ioc,
                        'ioc_type': event_ioc_type,
                        'timestamp': event_timestamp,
                        'incident_id': event_id,
                        'incident_name': COLLECTIVE_INSIGHTS_INCIDENT_NAME,
                        'incident_type': COLLECTIVE_INSIGHTS_INCIDENT_TYPE,
                        'detection_type': 'correlation',
                    }

                    if mitre_codes is not None:
                        insight_data['mitre_codes'] = self.parse_mitre_codes(mitre_codes)
                    if malwares is not None:
                        insight_data['malwares'] = [malwares]
                    rf_insight = RFInsight(**insight_data)
                    collective_insights_data.append(rf_insight)
            except KeyError as err:
                # don't kill if one bad event
                self.log.error(
                    f'Not able to parse Exabeam event for Collective Insights data: {err}'
                )

        return collective_insights_data

    def parse_mitre_codes(self, mitre_codes) -> list:
        """Parse Exabeam MITRE input and return list of T-codes

        Args:
            mitre_codes (list): List of MITRE codes from Exabeam event

        Returns:
            result (list): list of MITRE T-codes formatted for Recorded Future Collective Insights
        """
        return [mitre for mitre in mitre_codes if re.search(MITRE_REGEX, mitre)]

    def get_category(self, value) -> str:
        """Return the category of value (ip, domain...).

        value    - the value used in the patterns.
        """
        if re.match(r'^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$', value) is not None:
            return 'ip'

        if re.match(IPV6_REGEX, value) is not None:
            return 'ip'

        if re.match(r'^[a-z-.+]+:.+$', value) is not None:
            return 'url'

        if re.match(r'^\S+(\.\S+){1,}$', value) is not None:
            return 'domain'

        if re.match(r'(?i)^[0-9a-f]+$', value) is not None:
            return 'hash'

        if re.match(r'CVE-\d{4}-\d{4,7}', value) is not None:
            return 'vulnerability'

        raise CategoryParseException(f'Failed to automatically detect category for {value}')

    def write_latest_timestamp(self, timestamp_file: str = TIMESTAMP_FILE):
        """Writes timestamp from the latest Elastic alert to file if exists,
        othewise uses the script execution time

        Args:
            latest_timestamp (str): Timestamp from latest alert or execution time if no alerts
        """
        self.log.info(f"Writing latest timestamp to '{timestamp_file}'")
        config_dir = os.path.split(timestamp_file)[0]
        if not os.path.isdir(config_dir):
            self.log.info(f'Directory does not exist. Creating new {config_dir} directory')
            try:
                os.mkdir(path=config_dir)
            except ValueError as err:
                raise WriteFileError(str(err))
            except WriteFileError as err:
                raise WriteFileError(f'Directory {config_dir} is not writable: {err}')
        try:
            with open(timestamp_file, 'w') as f:
                f.write(self.latest_timestamp)
        except OSError as err:
            raise WriteFileError(str(err))
