"""
##################################### TERMS OF USE ###########################################
# The following code is provided for demonstration purpose only, and should not              #
# be used without independent verification. Recorded Future makes no representations         #
# or warranties, express, implied, statutory, or otherwise, regarding any aspect of          #
# this code or of the information it may retrieve, and provides it both strictly “as-is”     #
# and without assuming responsibility for any information it may retrieve. Recorded Future   #
# shall not be liable for, and you assume all risk of using, the foregoing. By using this    #
# code, Customer represents that it is solely responsible for having all necessary licenses, #
# permissions, rights, and/or consents to connect to third party APIs, and that it is solely #
# responsible for having all necessary licenses, permissions, rights, and/or consents to     #
# any data accessed from any third party API.                                                #
##############################################################################################
"""

import os
import sys
import json
import copy
import logging
import requests
import argparse
import ipaddress
from psengine.logger import RFLogger
from datetime import datetime, timedelta
from requests import ConnectTimeout, ConnectionError, HTTPError, ReadTimeout

APP_ID = "LogRhythm-Collective_Insights"
APP_VERSION = "1.0.0"
COLLECTIVE_INSIGHTS = "https://api.recordedfuture.com/collective-insights/detections"

logger = RFLogger().get_logger()


class LogRhythmError(Exception):
    """Error raised when there is an issue with the LogRhythm API."""

    pass


class RecordedFutureAPIError(Exception):
    """Error raised when there is an issue with the Recorded Future API."""

    pass


class RFScriptError(Exception):
    """Base class for other exceptions raised by this script."""

    pass


def get_args():
    """
    Defines the arguments the script can have on invocation.

    Returns:
        dict: dict with the arguments and their values
    """
    parser = argparse.ArgumentParser(
        description="LogRhythm Collective Insights Integration",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
    )
    parser.add_argument(
        "-rk",
        "--rfkey",
        dest="rf_api_key",
        help="Recorded Future API Key",
        default=os.environ.get("RF_API_KEY"),
        required=False,
    )
    parser.add_argument(
        "-lk",
        "--lrkey",
        dest="lr_api_key",
        help="LogRhythm API Key",
        default=os.environ.get("LOGRHYTHM_API_KEY"),
        required=False,
    )
    parser.add_argument(
        "-u",
        "--url",
        dest="lr_api_url",
        help="LogRhythm API URL. Example: http://localhost:8505",
        default=os.environ.get("LOGRHYTHM_API_URL"),
        required=False,
    )
    parser.add_argument(
        "-lb",
        "--lookback",
        dest="lookback",
        help="Number of hours to serach back for Alarms",
        default=1,
        required=False,
    )
    parser.add_argument(
        "-rf",
        "--rfonly",
        action="store_true",
        dest="rf_only",
        help="Only submit Recorded Future Alarms",
        required=False,
    )
    parser.add_argument(
        "-rbp",
        "--riskbasedpriority",
        dest="risk_based_priority",
        help="Threshold for RBP values for Alarms to submit",
        type=int,
        default=0,
        required=False,
    )
    parser.set_defaults(rf_only=False)
    parser.add_argument(
        "-d",
        "--debug",
        action="store_true",
        dest="debug_flag",
        help="Debug mode, insights not posted to Recorded Future",
        required=False,
    )
    parser.set_defaults(debug=False)
    parser.add_argument(
        "-l",
        "--loglevel",
        help="Log level",
        type=str,
        choices=["INFO", "WARNING", "ERROR", "CRITICAL", "DEBUG"],
        default="INFO",
        dest="loglevel",
        required=False,
    )

    parser.add_argument(
        "-ef",
        "--exclusion-file",
        help="Text file containing Alarm Rule names to be excluded from the Collective Insights results.",
        dest="exclusion_file",
        default="",
        required=False,
    )

    return parser.parse_args()


def lookup_alarm_events(lr_api_key, lr_api_url, alarm_search_details, rbp):
    """
    Queries LogRhythm API for Alarm Events based on the Alarm ID.

    Args:
        lr_api_key (str): LogRhythm
        lr_api_url (str): LogRhythm API URL
        alarm_search_details (list): list of Alarms to pull events for
        rbp (int): Risk Based Priority threshold for Alarms

    Returns:
        list: list of dicts of individual alarm events
    """
    ALARMS_LOOKUP_URL = f"{lr_api_url}/lr-alarm-api/alarms/"

    headers = {
        "Authorization": f"Bearer {lr_api_key}",
        "Content-Type": "application/json",
        "Accept": "application/json",
    }

    alarm_events = []

    try:
        logger.info("Querying LogRhythm for individual Alarm Events")
        for alarm in alarm_search_details:
            alarm_id = alarm["alarmId"]
            resp = requests.get(
                f"{ALARMS_LOOKUP_URL}/{alarm_id}/events", headers=headers
            )
            resp.raise_for_status()

            alarm_lookup = resp.json()
            alarm_lookup_events = alarm_lookup["alarmEventsDetails"]
            logger.debug(
                f"Alarm: {alarm_id} has the following events: {alarm_lookup_events}"
            )

            # Multiple events can be associated with a single Alarm, we want to submit an detection for each event
            for event in alarm_lookup["alarmEventsDetails"]:
                merged_alarm_dict = {**alarm, **event}
                alarm_events.append(merged_alarm_dict)
    except (HTTPError, ConnectTimeout, ConnectionError, ReadTimeout) as lre:
        logger.error(f"Error querying LogRhythm: {lre}")
        raise LogRhythmError(lre)

    # Final filtering for RBP
    logger.info(f"Filtering Alarm Events >={rbp} RBP")
    filtered_alarm_events = copy.deepcopy(alarm_events)
    try:
        for event in alarm_events:
            if event["priority"] < rbp:
                filtered_alarm_events.remove(event)
                logger.debug(
                    "One of the Events in Alarm {} has RBP {} which is lower than threshold, dropping from Events".format(
                        event["alarmId"], rbp
                    )
                )
    except Exception as rfe:
        logger.debug(
            f"Error encountered while filtering Alarm Events based on RBP: {rfe}"
        )
        raise RFScriptError(rfe)

    return filtered_alarm_events


def query_lr(lr_api_key, lr_api_url, lookback, rbp, rf_only, exclusions):
    """
    Queries LogRhythm API for Alarms that have been generated based on the lookback period
      and exclusions.

    Args:
        lr_api_key (str): LogRhythm API Key
        lr_api_url (str): LogRhythm API URL
        lookback (int): Number of hours back to search for Alarms
        rbp (int): Risk Based Priority threshold for alarms
        rf_only (bool): Whether to include only Recorded Future Alarms
        exclusions (list): list of Alarm names to be excluded from the results
          (OPTIONAL)

    Returns:
        list: write_back_data - list of json objects with incident data
        int: search_count - number of results found
    """
    ALARMS_SEARCH_URL = f"{lr_api_url}/lr-alarm-api/alarms"

    start_date = datetime.now()
    target_date = start_date - timedelta(hours=lookback)
    datetime_str = target_date.strftime("%Y-%m-%dT%H:%M:%S.%f")[:-3]

    headers = {
        "Authorization": f"Bearer {lr_api_key}",
        "Content-Type": "application/json",
        "Accept": "application/json",
    }

    params = {
        "orderby": "DateInserted",
        "dir": "descending",
        "dateInserted": datetime_str,
    }

    try:
        logger.info(
            f"Querying LogRhythm for Alarms inserted on or after {datetime_str}"
        )
        resp = requests.get(ALARMS_SEARCH_URL, headers=headers, params=params)
        resp.raise_for_status()
    except (HTTPError, ConnectTimeout, ConnectionError, ReadTimeout) as lre:
        logger.error(f"Error querying LogRhythm: {lre}")
        raise LogRhythmError(lre)

    alarm_search = resp.json()
    alarm_count = alarm_search["alarmsCount"]
    alarm_search_details = alarm_search["alarmsSearchDetails"]
    logger.info(f"Found {alarm_count} Alarms from initial search")

    if rf_only:
        logger.info("rf_only flag enabled, including only Recorded Future Alarms")
        alarm_search_details = [
            alarm
            for alarm in alarm_search_details
            if "Recorded Future" in alarm["alarmRuleName"]
        ]
    if exclusions:
        logger.info(f"exclusions list is not empty, filtering out exclusions: {exclusions}")
        alarm_search_details = [
            alarm
            for alarm in alarm_search_details
            if alarm["alarmRuleName"] not in exclusions
        ]

    logger.debug(f"Alarm search details: {alarm_search_details}")

    if not alarm_search_details:
        alarm_search_details = []

    alarm_events = lookup_alarm_events(
        lr_api_key, lr_api_url, alarm_search_details, rbp
    )

    write_back_data = []

    if alarm_events:
        # Alarm events have no threat_match_value, so we will submit insights for all possible IOCs
        # These submissions will share the same incident ID
        for event in alarm_events:
            if event["impactedHost"]:
                payload = create_collective_insights_payload(event, "impactedHost")
                write_back_data.append(payload)
            if event["impactedIP"]:
                payload = create_collective_insights_payload(event, "impactedIP")
                write_back_data.append(payload)
            if event["originHost"]:
                payload = create_collective_insights_payload(event, "originHost")
                write_back_data.append(payload)
            if event["originIP"]:
                payload = create_collective_insights_payload(event, "originIP")
                write_back_data.append(payload)
            if event["domain"]:
                payload = create_collective_insights_payload(event, "domain")
                write_back_data.append(payload)
            if event["url"]:
                payload = create_collective_insights_payload(event, "url")
                write_back_data.append(payload)
            if event["cve"]:
                payload = create_collective_insights_payload(event, "cve")
                write_back_data.append(payload)

        search_count = len(write_back_data)
    else:
        logger.info("No results found")
        search_count = 0

    return write_back_data, search_count


def create_collective_insights_payload(alarm_event, field):
    """
    Creates the payload to be submitted to Recorded Future
      Collective Insights API.

    Args:
        alarm_event (dict): dictionary of Alarm data to be uploaded
        field (str): field of the Alarm that contains the IOC

    Returns:
        dict: payload dictionary to be submitted to Recorded Future
    """
    type_mappings = {
        "impactedHost": "ip",
        "impactedIP": "ip",
        "originHost": "ip",
        "originIP": "ip",
        "domain": "domain",
        "url": "url",
        "cve": "vulnerability",
    }

    alarm_id = alarm_event["alarmId"]
    alarm_rule_name = alarm_event["alarmRuleName"]

    payload = {
        "timestamp": alarm_event["dateInserted"],
        "ioc": {
            "type": type_mappings[field],
            "value": alarm_event[field],
        },
        "incident": {
            "id": alarm_event["alarmId"],
            "name": f"{alarm_id}: {alarm_rule_name}",
            "type": "logrhythm-alarm",
        },
        "detection": {
            "name": f"{alarm_id}: {alarm_rule_name}",
            "type": "correlation",
            "sub_type": "",
        },
    }

    return payload


def submit_collective_insights(collective_insights_upload, rf_api_key):
    """
    submit_collective_insights submits data to Recorded Future Collective Insights API.

    Args:
        collective_insights_upload (list): list of dictionaries containing data to be uploaded
          to Recorded Future Collective Insights API
        rf_api_key (str): Recorded Future API Key

    Returns:
        dict : response - json response from Recorded Future Collective Insights API
    """
    headers = {
        "User-Agent": f"{APP_ID}/{APP_VERSION}",
        "Content-Type": "application/json",
        "Accept": "application/json",
        "X-RFToken": f"{rf_api_key}",
    }

    filtered_collective_insights_upload = copy.deepcopy(collective_insights_upload)

    # Filter out RFC1918 addresses from the ranges below
    private_ranges = [
        ipaddress.IPv4Network("10.0.0.0/8"),
        ipaddress.IPv4Network("172.16.0.0/12"),
        ipaddress.IPv4Network("192.168.0.0/16"),
    ]

    logger.debug(
        "Removing private IP addresses from the collective insights submission"
    )

    for insight in collective_insights_upload["data"]:
        if insight["ioc"]["type"] == "ip":
            try:
                ip = ipaddress.ip_address(insight["ioc"]["value"])
                if any(ip in network for network in private_ranges):
                    filtered_collective_insights_upload["data"].remove(insight)
            except ValueError:
                logger.debug(
                    f"Invalid IP address: {insight['ioc']['value']}, will not submit"
                )
                filtered_collective_insights_upload["data"].remove(insight)

    # Assign data variable to the json payload submitted to this function
    data = filtered_collective_insights_upload
    logger.info(
        "Generated {} individual detections to submit".format(
            len(filtered_collective_insights_upload["data"])
        )
    )
    logger.debug(f"Submitting data to Recorded Future: {data}")

    # Post data to collective insights using arguments created above
    try:
        response = requests.post(
            COLLECTIVE_INSIGHTS, headers=headers, data=json.dumps(data)
        )
        response.raise_for_status()
    except (HTTPError, ConnectTimeout, ConnectionError, ReadTimeout) as err:
        logger.error(f"Error submitting collective insights: {err}")
        raise RecordedFutureAPIError(err)

    return response


def get_collective_insights_exclusions(filename):
    """
    Reads the file inputted and returns a list of Alarm names to be
      excluded from the results.

    Args:
        filename (str): name of the file containing the exclusions

    Returns:
        list: list of Alarm names to be excluded from the results
    """

    exclusions_list = []

    exclusion_file = os.path.join(os.path.dirname(__file__), filename)

    if os.path.isfile(exclusion_file):
        try:
            with open(exclusion_file, "r") as file:
                try:
                    exclusions_list = file.readlines()
                except Exception as e:
                    logger.error(
                        f"""Error reading exclusions file: {exclusion_file}, continuing without exclusions: {e}"""
                    )
        except (FileNotFoundError, OSError) as fe:
            logger.error(
                f"Error reading exclusions file: {exclusion_file}, continuing without exclusions: {fe}"
            )
    else:
        logger.error(
            f"Error reading exclusions file: {exclusion_file}, continuing without exclusions..."
        )

    return exclusions_list


def main():
    """Main function that calls the other functions to query LogRhythm and submit the results
    to Recorded Future Collective Insights API."""

    try:
        args = get_args()

        if any(x is None for x in args.__dict__.values()):
            logger.error("Args not found. Exiting program...")
            raise RFScriptError("Args not found. Exiting program...")

    except (argparse.ArgumentError, RFScriptError):
        logger.error("Error parsing arguments", exc_info=True)
        sys.exit(1)

    json_options = {"options": {"debug": args.debug_flag, "summary": True}}
    logger.level = logging.getLevelName(args.loglevel)

    logger.info("Starting LogRhythm Collective Insights Integration")

    if args.exclusion_file != "":
        exclusions = get_collective_insights_exclusions(args.exclusion_file)
    else:
        exclusions = []

    try:
        args.lr_api_url = args.lr_api_url.rstrip("/")

        write_back_data, search_count = query_lr(
            args.lr_api_key,
            args.lr_api_url,
            int(args.lookback),
            args.risk_based_priority,
            args.rf_only,
            exclusions,
        )

        payload = dict(json_options, data=write_back_data)

        if search_count > 0:
            try:
                submit_payload = submit_collective_insights(payload, args.rf_api_key)
                logger.info(submit_payload.text)
                logger.info("Ending LogRhythm Collective Insights Integration")
            except (HTTPError, ConnectTimeout, ConnectionError, ReadTimeout):
                logger.error("Error submitting collective insights", exc_info=True)
                sys.exit(1)
        else:
            logger.info("No results to upload to Recorded Future Collective Insights.")
    except LogRhythmError:
        logger.error(f"Error querying LogRhythm", exc_info=True)
        sys.exit(1)


if __name__ == "__main__":
    main()
