"""##################################### TERMS OF USE ###########################################
# The following code is provided for demonstration purpose only, and should not              #
# be used without independent verification. Recorded Future makes no representations         #
# or warranties, express, implied, statutory, or otherwise, regarding any aspect of          #
# this code or of the information it may retrieve, and provides it both strictly “as-is”     #
# and without assuming responsibility for any information it may retrieve. Recorded Future   #
# shall not be liable for, and you assume all risk of using, the foregoing. By using this    #
# code, Customer represents that it is solely responsible for having all necessary licenses, #
# permissions, rights, and/or consents to connect to third party APIs, and that it is solely #
# responsible for having all necessary licenses, permissions, rights, and/or consents to     #
# any data accessed from any third party API.                                                #
##############################################################################################
"""

import json
import os
import sys
from argparse import ArgumentDefaultsHelpFormatter, ArgumentParser

from psengine.collective_insights import CollectiveInsightsError, RFCollectiveInsights
from psengine.config import Config, ConfigError
from psengine.logger import RFLogger

from libexabeam._version import APP_VERSION
from libexabeam.constants import APP_ID
from libexabeam.errors import (
    CategoryParseException,
    ExabeamError,
    RFScriptError,
    WriteFileError,
)
from libexabeam.exabeam_ci import ExabeamCI

LOG = RFLogger().get_logger()


def get_args() -> ArgumentParser:
    """Gets arguments from the command line.

    Returns:
        dict: dict with the arguments and their values
    """
    parser = ArgumentParser(
        description='Exabeam Collective Insights Integration',
        formatter_class=ArgumentDefaultsHelpFormatter,
    )
    parser.add_argument(
        '-k',
        '--key',
        dest='rf_api_key',
        help='Recorded Future API Key',
        default=os.environ.get('RF_API_KEY'),
        required=False,
    )
    parser.add_argument(
        '-c',
        '--client-id',
        dest='client_id',
        help='Exabeam Client API Key',
        default=os.environ.get('EXABEAM_CLIENT_ID'),
        required=False,
    )
    parser.add_argument(
        '-s',
        '--client-secret',
        dest='client_secret',
        help='Exabeam Client API Secret',
        default=os.environ.get('EXABEAM_CLIENT_SECRET'),
        required=False,
    )
    parser.add_argument(
        '-r',
        '--region',
        dest='region',
        help='Exabeam Instance Region',
        choices=['us-west', 'us-east', 'canada', 'europe', 'singapore', 'japan', 'australia'],
        default=os.environ.get('REGION'),
        required=False,
    )
    parser.add_argument(
        '-L',
        '--lookback',
        dest='lookback',
        help='Lookback time for detections in hours, defaults to 1',
        default=1,
        type=int,
        required=False,
    )
    parser.add_argument(
        '--debug',
        action='store_true',
        dest='debug_flag',
        help='Debug mode, insights not posted to Recorded Future',
        required=False,
    )
    parser.add_argument(
        '-l',
        '--loglevel',
        help='Log level',
        type=str,
        choices=['DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL'],
        default='INFO',
        dest='log_level',
        required=False,
    )

    return parser.parse_args()


def main():
    """Main function that calls the other functions to fetch Exabeam alerts
    and submit the results to Recorded Future Collective Insights API.
    """
    LOG.info(f'Recorded Future Exabeam Collective Insights v{APP_VERSION} starting')

    try:
        args = get_args()
        LOG.setLevel(args.log_level)
        config = Config()
        config.rf_token = args.rf_api_key
        config.app_id = f'{APP_ID}/{APP_VERSION}'
        ci_mgr = RFCollectiveInsights(cfg=config)
        exabeam_ci_client = ExabeamCI(
            region=args.region,
            client_id=args.client_id,
            client_secret=args.client_secret,
            lookback=args.lookback,
        )
    except (ConfigError, ValueError, ExabeamError, RFScriptError) as err:
        LOG.critical(err, exc_info=False)
        sys.exit(1)

    # Get Exabeam Events and Collective Insights context
    try:
        LOG.info('Fetching Events from Exabeam')
        exabeam_events = exabeam_ci_client.fetch_exabeam_events()
    except ExabeamError as err:
        LOG.error(f'Failed to fetch alerts from Exabeam. Cause: {err}', exc_info=False)
        sys.exit(1)

    if exabeam_events:
        LOG.info(f'Retrieved {len(exabeam_events)} events from Exabeam')
        LOG.debug(f'Alerts retrieved: {json.dumps(exabeam_events)}')
        # Get each Exabeam Event IOC details
        try:
            LOG.info('Fetching Event IOC fields from Exabeam')
            exabeam_event_details = exabeam_ci_client.fetch_exabeam_event_details(
                exabeam_events=exabeam_events
            )
            LOG.debug(f'Exabeam Events with IOC Fields: {exabeam_event_details}')
        except ExabeamError as err:
            LOG.error(f'Unable to fetch Exabeam event details: {err}')

        # Format and send payload to Recorded Future Collective Insights
        try:
            collective_insights_payload = exabeam_ci_client.format_exabeam_events(
                exabeam_events=exabeam_event_details
            )
            response = ci_mgr.submit(insight=collective_insights_payload, debug=args.debug_flag)
            LOG.debug(f'Collective Insights API Response: {response}')
        except CategoryParseException as err:
            LOG.error(f'Unable to parse indicator type from Exabeam event: {err}')
        except ExabeamError as err:
            LOG.error(f'Unable to format Exabeam events into Collective Insights payload: {err}')
        except CollectiveInsightsError as err:
            LOG.error(f'Submission to Collective Insights API failed. Cause: {err}', exc_info=False)

    else:
        LOG.info('No Exabeam events to upload to Recorded Future Collective Insights')

    try:
        exabeam_ci_client.write_latest_timestamp()
    except WriteFileError as err:
        LOG.critical(f'Failed to write latest timestamp to file. Cause: {err}', exc_info=False)

    LOG.info('Recorded Future Exabeam Collective Insights completed')


if __name__ == '__main__':
    main()
