"""##################################### TERMS OF USE ###########################################
# The following code is provided for demonstration purpose only, and should not              #
# be used without independent verification. Recorded Future makes no representations         #
# or warranties, express, implied, statutory, or otherwise, regarding any aspect of          #
# this code or of the information it may retrieve, and provides it both strictly “as-is”     #
# and without assuming responsibility for any information it may retrieve. Recorded Future   #
# shall not be liable for, and you assume all risk of using, the foregoing. By using this    #
# code, Customer represents that it is solely responsible for having all necessary licenses, #
# permissions, rights, and/or consents to connect to third party APIs, and that it is solely #
# responsible for having all necessary licenses, permissions, rights, and/or consents to     #
# any data accessed from any third party API.                                                #
##############################################################################################
"""

import json
import os
import sys
import warnings
from argparse import ArgumentDefaultsHelpFormatter, ArgumentParser

from psengine.collective_insights import CollectiveInsightsError, RFCollectiveInsights
from psengine.config import Config, ConfigError
from psengine.logger import RFLogger
from urllib3.exceptions import InsecureRequestWarning

from libelastic._version import APP_VERSION
from libelastic.constants import HOSTNAME, LOOKBACK, TIMESTAMP
from libelastic.errors import ElasticClientError, WriteFileError
from libelastic.es_ci import ESCI

# Suppress only the InsecureRequestWarning from urllib3
warnings.simplefilter('ignore', InsecureRequestWarning)

APP_ID = 'elk-ci-partner'

LOG = RFLogger().get_logger()

SETTINGS_FILE = os.path.join('config', 'settings.ini')


def get_cmdline_args() -> ArgumentParser:
    """Gets arguments from the command line.

    Returns:
        dict: dict with the arguments and their values
    """
    parser = ArgumentParser(
        description='Elastic Collective Insights Integration',
        formatter_class=ArgumentDefaultsHelpFormatter,
    )
    parser.add_argument(
        '-k',
        '--key',
        dest='rf_api_key',
        help='Recorded Future API Key',
        default=os.environ.get('RF_API_KEY'),
        required=False,
    )
    parser.add_argument(
        '-u',
        '--user',
        dest='elastic_user',
        help='Elasticsearch user',
        default=os.environ.get('ELASTIC_USER'),
        required=False,
    )
    parser.add_argument(
        '-p',
        '--password',
        dest='elastic_password',
        help='Elasticsearch password',
        default=os.environ.get('ELASTIC_PASSWORD'),
        required=False,
    )
    parser.add_argument(
        '-t',
        '--timestamp',
        dest='timestamp',
        help='Elastic timestamp field.',
        default=TIMESTAMP,
        required=False,
    )
    parser.add_argument(
        '--hostname',
        dest='hostname',
        help='Elastic hostname field',
        default=HOSTNAME,
        required=False,
    )
    parser.add_argument(
        '-L',
        '--lookback',
        dest='lookback',
        help='Initial lookback time for detections in hours',
        default=LOOKBACK,
        type=int,
        required=False,
    )
    parser.add_argument(
        '--debug',
        action='store_true',
        dest='debug_flag',
        help='Debug mode, insights not posted to Recorded Future',
        required=False,
    )
    parser.add_argument(
        '-l',
        '--log-level',
        help='Loging level of the integration script',
        type=str,
        choices=['DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL'],
        default='INFO',
        dest='log_level',
        required=False,
    )
    return parser.parse_args()


def main():
    """Main function that calls the other functions to fetch elastic alerts
    and submit the results to Recorded Future Collective Insights API.
    """
    LOG.info(f'Recorded Future Elasticsearch Collective Insights v{APP_VERSION} starting')

    try:
        args = get_cmdline_args()
        LOG.setLevel(args.log_level)

        config = Config(SETTINGS_FILE)
        config.rf_token = args.rf_api_key
        config.app_id = f'{APP_ID}/{APP_VERSION}'
        ci_mgr = RFCollectiveInsights(cfg=config)

        esci = ESCI(
            config=config,
            elastic_user=args.elastic_user,
            elastic_pass=args.elastic_password,
            lookback=args.lookback,
            timestamp_field=args.timestamp,
            hostname=args.hostname,
        )
    except (ConfigError, ValueError, ElasticClientError) as ce:
        LOG.critical(ce, exc_info=False)
        sys.exit(1)

    # Get elastic Events and Collective Insights context
    try:
        LOG.info('Fetching alerts from Elasticsearch')
        elastic_alert_response = esci.fetch_elastic_alerts()
    except ElasticClientError as err:
        LOG.error(f'Failed to fetch alerts from Elasticsearch. Cause: {err}', exc_info=False)
        sys.exit(1)

    elastic_alerts = elastic_alert_response.get('hits', {}).get('hits')
    if elastic_alerts:
        LOG.info(f'Retrieved {len(elastic_alerts)} alerts from Elasticsearch')
        LOG.debug(f'Alerts retrieved: {json.dumps(elastic_alerts)}')

        # Format and send CI payload to Collective Insights
        ci_payload = esci.format_ci_payload(elastic_alerts=elastic_alerts)
        try:
            response = ci_mgr.submit(insight=ci_payload, debug=args.debug_flag)
            LOG.debug(f'Collective Insights API Response: {response}')

        except (TypeError, ValueError) as err:
            LOG.error('Supplied Collective Insights paramaters are invalid. Exiting')
            LOG.debug(err)
            sys.exit(1)

        except CollectiveInsightsError as err:
            LOG.error(
                f'Submission to Collective Insights API failed. Exiting. Cause: {err}',
                exc_info=False,
            )
            sys.exit(1)
    else:
        LOG.info('No Elasticsearch alerts found to submit to Collective Insights')

    try:
        esci.write_latest_timestamp()
    except WriteFileError as err:
        LOG.critical(f'Failed to write latest timestamp to file. Cause: {err}', exc_info=False)

    LOG.info('Recorded Future Elasticsearch Collective Insights completed')


if __name__ == '__main__':
    main()
